SUBROUTINE lsq_fit_2 ( x, y, nvals, c, error )
!
!  Purpose:
!    To perform a least-squares fit of an input data set
!    to the parabola
!        y(x) = c(0) + c(1) * x + c(2) * x**2,
!    and return the resulting coefficients.  The input
!    data set consists of nvals (x,y) pairs contained in  
!    arrays x and y.  The output coefficients of the
!    quadratic fit c0, c1, and c2 are placed in array c. 
!
!  Record of revisions:
!      Date       Programmer          Description of change
!      ====       ==========          =====================
!    02/10/96    S. J. Chapman        Original code
!
IMPLICIT NONE

! Declare dummy arguments:
INTEGER, INTENT(IN) :: nvals             ! Number of input data pts
REAL, DIMENSION(nvals), INTENT(IN) :: x  ! Input x values
REAL, DIMENSION(nvals), INTENT(IN) :: y  ! Input y values
REAL, DIMENSION(0:2), INTENT(OUT) :: c   ! Coefficients of fit
INTEGER, INTENT(OUT) :: error            ! Error flag:
                                         !  0 = No error.
                                         !  1 = Singular eqns
                                         !  2 = Insufficient data

! Declare local variables:
REAL, DIMENSION(3,3) :: a        ! Coefficients of eqn to solve
REAL, DIMENSION(3) :: b          ! Right side of coefficient eqns
INTEGER :: i, j                  ! Index variable
REAL, DIMENSION(0:2) :: soln     ! Solution vector
REAL,DIMENSION(0:4) :: sum_xi    ! Sum of x**i values
REAL,DIMENSION(0:2) :: sum_xiy   ! Sum of x**i*y values

! First, check to make sure that we have enough input data.
check_data: IF ( nvals < 3 ) THEN
 
   ! Insufficient data.  Set error = 2, and get out.
   error = 2
ELSE 
 
   sum_xi = 0.                   ! Clear sums
   sum_xiy = 0.          

   ! Build the sums required to solve the equations.
   sums: DO j = 1, nvals
      DO i = 0, 4
         sum_xi(i) = sum_xi(i) + x(j)**i
      END DO
      DO i = 0, 2
         sum_xiy(i) = sum_xiy(i) + x(j)**i * y(j)
      END DO
   END DO sums
 
   ! Set up the coefficients of the equations.
   a(:,1) = sum_xi(0:2)
   a(:,2) = sum_xi(1:3)
   a(:,3) = sum_xi(2:4)
   b(:) = sum_xiy(0:2)

   ! Solve for the least squares fit coefficients.  They will 
   ! be returned in array b if error = 0.
   CALL simul ( a, b, soln, 3, 3, error )
 
   ! If error == 0, return the coefficients to the user.
   return: IF ( error == 0 ) THEN
      c = soln
   ELSE
      c = 0.
   END IF return
END IF check_data
 
END SUBROUTINE lsq_fit_2
