SUBROUTINE quad ( a, b, c, x1, x2, error )
!
!  Purpose:
!    To solve for the real roots of the quadratic equation
!          a*x**2 + b*x + c = 0
!    in a robust manner by normalizing the coefficients
!    of the equation and avoiding the necessity of 
!    subtracting two nearly equal numbers in the solution.
!
!  Record of revisions:
!      Date       Programmer          Description of change
!      ====       ==========          =====================
!    02/08/96    S. J. Chapman        Original code
!
IMPLICIT NONE

! Declare dummy arguments:
REAL, INTENT(IN) :: a            ! Coefficient of x**2
REAL, INTENT(IN) :: b            ! Coefficient of x
REAL, INTENT(IN) :: c            ! Constant coefficient
REAL, INTENT(OUT) :: x1          ! Root 1
REAL, INTENT(OUT) :: x2          ! Root 2
INTEGER, INTENT(OUT) :: error    ! Error flag: 0=no error
                                 !             1=complex roots

! Declare local variables
REAL :: an                  ! Normalized coefficient of x**2
REAL :: big                 ! Abs value of largest coef.
REAL :: bn                  ! Normalized coefficient of x
REAL :: cn                  ! Normalized constant coefficient
REAL :: discriminant        ! Discriminant of equation

! Normalize coefficients
big = MAX ( ABS(a), ABS(b), ABS(c) )
an = a / big
bn = b / big
cn = c / big

! Calculate discriminant
discriminant = bn**2 - 4.*an*cn

! Get roots
IF ( discriminant >= 0 ) THEN
   IF ( -bn > 0 ) THEN
      x1 = ( -bn + SQRT(discriminant) ) / (2. * an)
   ELSE
      x1 = ( -bn - SQRT(discriminant) ) / (2. * an)
   END IF
   x2 = cn / ( an * x1 )
   error = 0
ELSE
   error = 1
END IF

END SUBROUTINE quad
