require File.dirname(__FILE__) + '/../test_helper'
require 'mocha'

class UsersControllerTest < Test::Rails::ControllerTestCase

  fixtures :topics, :users, :settings, :support_materials, :readings
  
  def setup
    super
    login_as :laura
  end
  
  def test_show_new
    get 'new'
    assert_response :success
    assert_assigned :user    
  end
  
  def test_create_expert
    assert_difference(Expert, :count) do
      post_new_expert
    end
  end
  
  def test_create_expert_sets_flash
    post_new_expert
    assert flash[:notice]
  end
  
  def test_create_expert_redirect
    post_new_expert
    assert_response :redirect
  end

  # only admin can access the expert controller
  def test_redirect_participant
    login_as :ernest # a participant
    get :index
    assert_response :redirect
  end

  def test_redirect_expert
    login_as :frank # an editor
    get :index
    assert_response :redirect
  end
  
  def test_permit_admin
    login_as :laura # an admin
    get :index
    assert_response :success
  end

  
  def test_index_shows_users
    get :index
    assert_assigned :users
  end
  
  def test_update_not_explosive
    assert_nothing_raised { update_user }
  end
  
  def test_edit_should_redirect
    update_user
    assert_response :redirect
  end
  
  def test_edit_should_set_flash
    update_user
    assert !flash.empty?
    assert flash[:notice]
    assert_equal "Expert Account Updated", flash[:notice]
  end
  
  def test_show_should_set_user_and_commented_topics
    ApplicationPolicy.stubs(:can_send_message_to?).returns(true)
    get :show, :id => 3
    assert_assigned :profile
    assert_assigned :commented_topics
  end
  
  def test_show_accessible_by_anyone
    @request.session[:user] = nil
    get :show, :id => 3
    assert_response :success
  end
  
  def test_show_should_not_populate_login_form
    @request.session[:user] = nil
    get :show, :id => 3
    assert !@response.body.match(User.find(3).email) # email attribute of signup form
  end
  
  [[1, :particpant], [3, :expert], [5, :admin]].each do |id, role|
    define_method("test_show_#{role}") do
      @request.session[:user] = id
      assert_nothing_raised { get :show, :id => id }
    end
  end

  def test_show_reveals_avatar_form_to_self
    @request.session[:user] = 1
    get :show, :id => 1
    assert_match "<a href=\"/avatars/new\">", @response.body    # should be in a view test     
    # assert_match "<form action=\"/avatars/new\" method=\"get\">", @response.body    # should be in a view test     
  end
  
  def test_create_participant_creates_new
    assert_difference(Participant, :count) { post_new_participant }
  end
  
  def test_create_participant_with_zip_code
    assert_difference(Participant, :count) { post_new_participant(:postal_code => '902') }
  end
  
  # TODO view test Brief Bio shows only for Experts
 private

  def update_user
    post :edit, {
      "commit"=>"Save this Expert Account", 
      "id"=>"3", 
      "user"=>{
        "name"=>"Frank", 
        "password_confirmation"=>"silly", 
        "picture"=>"",
        "bio"=>"Sign up now to be instantly ", 
        "password"=>"silly", 
        "email"=>"frank@omara.ca"
      }
    }
  end

  def post_new_participant(changes={})
    post :new, 
      :type => 'participant', 
      :user => {
        :name => 'Peter', 
        :email => 'p.rick@shamrock.org', 
        :password => 'clover', 
        :password_confirmation => 'clover',
        :postal_code => 'K1R',
        :gender => 1,
        :year_of_birth => '1970'
        }.merge(changes)
  end

  def post_new_expert
    post :new, :type => 'expert', :user => {:name => 'Edward', :email => 'ed@test.com', :password => 'expert', :password_confirmation => 'expert', :bio => 'Expert he is'}
  end

end
