;
; Create an Apr-Sep mean timeseries of gridded IST from monthly observed IST.
; Only the northern hemisphere is kept.  Also, a land-sea mask is read
; in and the northerm hemisphere extracted.  Finally due to so much missing
; data I allow just 3 months of obs to be enough to get a AMJJAS mean, and then
; if any point is missing at a particular time but 3 or more surrounding
; points have values then it is infilled with the mean of those points.
; After this spatial infilling, year-by-year timeseries are considered.  Any
; missing values that have non-missing values before and after are infilled
; by a mean of those values.
; Also plots the long-term s.d.
;
; CAN OPTIONALLY COMPUTE THE JJA TEMP INSTEAD OF AMJJAS TEMP
; OR a ONDJFM
;
dojja=2        ; 0=Apr-Sep 1=Jun-Aug 2=Oct-Mar
doold=2        ; 0=use new data set (2002), 1=use old version, 2=use land-only!
;
print,'Reading in IPCC global IST gridded data'
case doold of
  0: begin
    fnin='/cru/u2/f055/data/obs/grid/surface/hadcrut_18562001.mon.nc'
    yren=2001
    fnout=''
  end
  1: begin
    fnin='/cru/u2/f055/data/obs/grid/surface/ist_ipcc_18561999.mon.nc'
    yren=1999
    fnout='..old..'
  end
  2: begin
    fnin='/cru/u2/f055/data/obs/grid/surface/crutem2_18512001.mon.nc'
    yren=2001
    fnout='..land..'
  end
endcase
;
print,fnin
ncid=ncdf_open(fnin)
fdmon=crunc_rddata(ncid,tst=[1856,0],tend=[yren,11],grid=g,info=i)
ncdf_close,ncid
;
; Compute Apr-Sep means from the data (requiring 3 or more months of data)
; or Jun-Aug means from the data (requiring 2 or more months of data)
; or Oct-Mar means from the data (requiring 3 or more months of data)
;
nmon=12
nyr=g.nt/nmon
fdmon=reform(fdmon,g.nx*g.ny,nmon,nyr)
case dojja of
  0: fdseas=mkseason(fdmon,3,8,datathresh=3)
  1: fdseas=mkseason(fdmon,5,7,datathresh=2)
  2: fdseas=mkseason(fdmon,9,2,datathresh=3)
endcase
fdseas=reform(fdseas,g.nx,g.ny,nyr)
timey=findgen(nyr)+g.year(0)
;
; Keep a copy of this raw seasonal data
;
rawseas=fdseas
;
xlon=g.x
ylat=g.y
nx=g.nx
ny=g.ny
;
; Now do some infilling if possible from surrounding values
;
print,'Smoothing',nyr
for i = 0 , nyr-1 do begin
  print,i
  onefd=reform(fdseas(*,*,i))
  fdinf=tim_infill(onefd,3.,/cyclic,thresh=0.42,ylat=ylat)
  fdseas(*,*,i)=fdinf(*,*)
endfor
;
; Now do some infilling if possible from previous/subsequent values
;
newfd=fdseas
for ix = 0 , nx-1 do begin
  for iy = 0 , ny-1 do begin
    print,ix,iy
    for iyr = 1 , nyr-2 do begin
      if finite(newfd(ix,iy,iyr)) eq 0 then begin
        newfd(ix,iy,iyr)=0.5*(fdseas(ix,iy,iyr-1)+fdseas(ix,iy,iyr+1))
      endif
    endfor
  endfor
endfor
fdseas=newfd
;
; Now read in the land/sea fraction (1=land 0=ocean)
;
lsm=fltarr(nx,ny)
openr,1,'/cru/u2/f055/detect/obsdat/landsea.dat'
readf,1,format='(72I6)',lsm
close,1
lsm=lsm/100.
landonly=lsm*!values.f_nan
landonly(where(lsm eq 1.))=1.
seaonly=lsm*!values.f_nan
seaonly(where(lsm eq 0.))=1.
mainlysea=lsm*!values.f_nan
mainlysea(where(lsm lt 0.25))=1.
;
; Just keep the northern hemisphere stuff
;
ykeep=where(ylat ge 0.,ny)
ylat=ylat(ykeep)
fdseas=fdseas(*,ykeep,*)
rawseas=rawseas(*,ykeep,*)
landonly=landonly(*,ykeep)
seaonly=seaonly(*,ykeep)
mainlysea=mainlysea(*,ykeep)
;
; Now compute long-term mean and s.d.
;
fdd=double(fdseas)
fdtot=total(fdd,3,/nan)
fdnum=float(total(finite(fdd),3))
fdltm=fdtot/fdnum
ml=where(fdnum lt 20.,nmiss)
if nmiss gt 0 then fdltm(ml)=!values.f_nan
;
fdsqu=total(fdd^2,3,/nan)
fdsqu=fdsqu/fdnum
fdsd=sqrt(fdsqu-fdltm^2)
;
; Now compute a year-by-year timeseries of the fraction of missing data
;
nall=total(finite(fdltm))
nmiss=float(total(finite(fdseas),1))
nmiss=1.-(total(nmiss,1)/nall)
missfrac=nmiss
;
case dojja of
  0: fn='obs_temp_as.idlsave'
  1: fn='obs_temp_jja.idlsave'
  2: fn='obs_temp_om.idlsave'
endcase
save,filename=fn+fnout,timey,fdseas,xlon,ylat,nyr,nx,ny,$
  fdltm,fdsd,landonly,seaonly,mainlysea,missfrac,rawseas
;
loadct,39
multi_plot,nrow=2
if !d.name eq 'X' then window,ysize=850
;
map=def_map(/npolar)  &  map.limit(0)=15.
coast=def_coast(/get_device)
labels=def_labels(/off)
;
case dojja of
  0: tit='Apr-Sep'
  1: tit='Jun-Aug'
  2: tit='Oct-Mar'
endcase
levels=findgen(17)*0.1
if dojja eq 2 then levels=levels*2.
inter_boxfd,fdsd,xlon,ylat,$
  /scale,$
  title='Long-term-standard deviation '+tit+' temperature',$
  map=map,coast=coast,labels=labels,$
  levels=levels
;
plot,timey,missfrac,psym=10,/xstyle,xtitle='Year',$
  ytitle='Fraction of data missing per season',yrange=[0.,1],/ystyle
;
end
