module CommentHelper
  
  # shows only the first line of a comment, with a link to reveal the rest
  def comment_teaser(comment)
    simple_format(h(comment.body))
  end

  # returns the rating widget, the reply link as well as the remove link(admin only)
  # in a div tag.
  def comment_tools(comment)
    content_tag(:div,
      rate_topic_form_and_paragraph(comment.ratings.for_user(current_user)) do
        [ reply_link(comment),
          remove_link(comment),
        ].compact.join(' ')        
      end ,
      {:class => "comment-tools", :id => "comment_tools_#{comment.id}"})
  end

  # returns the commenter's name and role as well as a comment permalink
  # in a pragraph.
  def comment_info(comment, include_permalink = true)
    content_tag(:p,
       [commenter_name(comment), 
        commenter_role(comment), 
        commenter_province(comment), 
        include_permalink ? comment_permalink(comment) : timestamp(comment)
        ].compact.join(' '),
      {:class => "comment-info"}
    )
  end

  # returns this comments author. If the current user is admin, the name will be an email link
  def commenter_name(comment)
    return '*unknown*' if comment.user.nil?
    if current_user && current_user.role == :admin
      link_to(comment.user.name, "mailto:#{comment.user.email}", {:title => 'email this user'})
    else
      comment.user.name
    end
  end

  # returns "moderator" for admins, and "expert" for experts
  def commenter_role(comment)
    return if comment.user.nil?
    case comment.user.role
      when :admin then "(moderator)"
      when :expert then 
        return "(the pollster)" if comment.user.name == 'Nik Nanos'
        "(expert)"
    end
  end  
  
  # returns the name of the commenters province, if found otherwise, returns nil
  def commenter_province(comment)
    "(%s)" % comment.user.province unless comment.user.province.blank?
  end

  def comment_permalink(comment)
    content_tag(:span,
      link_to( 
        timestamp(comment),
        {
          :controller => '/topics', 
          :action => 'show', 
          :id => comment.topic_id,  # PREMATURE to avoid hitting the DB again
          :anchor => "comment_#{comment.id}" 
        }, 
        :title => 'link to this comment'),
      {:class => "comment-permalink"}
    )
  end

  def reply_link(comment)
    link_to_remote(
      "Reply to Comment",
      {
        :url => { 
          :controller => 'talk', 
          :action => 'get_comment_form', 
          :topic_id => comment.topic_id, # PREMATURE
          :parent_id => comment.id
        },
        :before => 'CommentManager.beforeReplyForm(' + comment.id.to_s + ')'
      },
      {:class => 'reply',
       :id => "reply_#{comment.id}_link"})
  end

  def remove_link(comment)
    return unless current_user.role == :admin
    link_to_remote(
      "remove comment", 
      {
        :url => remove_comment_url(:id => comment)
        # :before => update_element_function(rating_id(rating), :position => :bottom, :content => '<li>busy</li>')
      }, 
      { # HTML options
        :href  => remove_comment_url(:id => comment),
        :title => "remove this comment", 
        :class => "remove",
        :id    => "remove_#{comment.id}"
      }
    )  
  end
  
  # returns a link to "Join this conversation" that, when clicked, toggles open the collapsed
  # conversation and replaces this link with a link labelled "Hide this conversation"
  def join_hide_conversation_toggle_link(thread_starter)
    link_to_function("Join this conversation", "CommentManager.showReplies('comment_#{thread_starter.id}_wrapper');",
          :class => 'conversation_toggle_join'
          ) +
    link_to_function("Hide this conversation", "CommentManager.hideReplies('comment_#{thread_starter.id}_wrapper');",
          :class => 'conversation_toggle_hide',
          :style => 'display: none'
          )
  end
  
  def hot_comment(comment)
    content_tag(:p, truncate(comment.body, 80, '') + 
    ( block_given? ? yield : hot_comment_link(comment) ) ) + 
    comment_info(comment, false)
  end
 
  def timestamp(comment)
    content_tag(:span,
        # comment.created_at.to_s(:short), # absolute reference
        # comment.created_at.gmtime.strftime("%m/%d/%Y %H:%M:%S"), # for use with JS (would be needed if page were cached)
        time_ago_in_words(comment.created_at) + ' ago', # use relative time (eg. "5 minutes ago")
      {:class => "comment-timestamp"}
    )
  end
  
  def hot_comment_link(comment)
    link_to_function "... more", "return CommentManager.jumpTo('comment_#{comment.id}');"
  end
  
  def comment_class(comment)
    comment.user.role rescue nil
  end
end