require File.dirname(__FILE__) + '/../test_helper'

class SupportMaterialTest < Test::Rails::TestCase
  fixtures :readings, :users, :support_materials, :ratings, :topics

  def test_readings_are_dependent_on_support_material
    assert_difference(Reading, :count, '', -1) do
      SupportMaterial.destroy(support_materials(:msn).id)
    end    
  end
  
  def test_ratings_are_dependent_on_support_material
    assert_difference(AbstractRating, :count, '', -2) do
      SupportMaterial.destroy(support_materials(:google).id)
    end    
  end

  def test_create_with_url
    assert_valid create_support_material()
  end

  def test_create_with_file_html
    m = create_support_material_html_file()
    assert_valid m
    assert_equal 'text/html', m.content_type, "content type is missing"
  end
  
  def test_create_with_file_pdf
    m = create_support_material_pdf_file()
    assert_valid m
    assert_equal 'application/pdf', m.content_type, "content type is missing"
  end

  def test_should_require_url_if_no_file
    material = create_support_material(:url => nil)
    assert material.errors.on_base
  end

  def test_should_require_file_if_no_url
    material = create_support_material_html_file(:uploaded_data => nil)
    assert !material.valid?
    assert material.errors.on(:size)
    assert material.errors.on(:content_type)
    assert material.errors.on(:filename)
  end
  
  def test_should_not_allow_both_file_and_url
    material = create_support_material_html_file(:url => 'http://sesresearch.com/who_voted.html')
    assert material.errors.on_base    
  end
  
  def test_should_require_file_to_be_html_or_pdf
    material = create_support_material({:url => nil, :uploaded_data => mock_upload('results.xl', 'application/excel') } )
    assert_equal 'application/excel', material.content_type
    assert !material.valid?
    assert material.errors.on('content_type')
  end
  
  def test_informative_ratings_for_user
    assert_kind_of MaterialInformativeRating, support_materials(:google).informative_ratings.for_user(users(:ernest))
  end
  
  def test_convincing_ratings_for_user
    assert_kind_of MaterialConvincingRating,  support_materials(:google).convincing_ratings.for_user(users(:ernest))
  end
    
  def test_informative_ratings_for_user_participant_already_rated_returns_existing_rating    
    rating = support_materials(:google).informative_ratings.for_user(users(:ernest))
    assert_equal AbstractRating.find(22), rating
    assert !rating.new_record?
  end
  
  def test_convincing_ratings_for_user_participant_already_rated_returns_existing_rating    
    rating = support_materials(:google).convincing_ratings.for_user(users(:ernest))
    assert_equal AbstractRating.find(24), rating
    assert !rating.new_record?
  end
  
  def test_informative_ratings_for_user_guest
    assert_kind_of MaterialInformativeRating, support_materials(:google).informative_ratings.for_user(users(:visitor1))
  end
  
  def test_informative_ratings_for_user_guest
    assert_kind_of MaterialConvincingRating, support_materials(:google).convincing_ratings.for_user(users(:visitor1))
  end
  
  def test_title
    assert_equal "link http://www.google.com", support_materials(:google).title
  end

  def test_title_file
    assert_equal "file somefile.html", support_materials(:uploaded_file).title
  end
  
  protected
  
  def create_support_material(options = {})
    SupportMaterial.create({ :topic_id => 1, :name => 'Who voted in 2006: SES Research, Ottawa, 2006', :url => 'http://sesresearch.com/who_voted.html' }.merge(options))
  end
  
  def create_support_material_html_file(options = {})
    create_support_material({:url => nil, :uploaded_data => mock_upload('voted2006.html', 'text/html')}.merge(options))
  end
  
  def create_support_material_pdf_file(options = {})
    create_support_material({:url => nil, :uploaded_data => mock_upload('design.pdf', 'application/pdf')}.merge(options))
  end
  
  def mock_upload(path, content_type)
    fixture_file_upload("files/" + path, content_type) 
  end
  
end
