# Schema as of Fri Jun 29 21:58:21 EDT 2007 (schema version 45)
#
#  id                  :integer(11)   not null
#  name                :string(255)   default(), not null
#  url                 :string(255)   default()
#  topic_id            :integer(11)   default(0), not null
#  filename            :string(255)   
#  reading_count       :integer(11)   
#  content_type        :string(255)   
#  size                :integer(11)   
#  parent_id           :integer(11)   
#

class SupportMaterial < ActiveRecord::Base  
  extend ReadingFinderExtension
  
  has_attachment :content_type => ['application/pdf', 'text/html'],
    :file_system_path => 'private', :max_size => 10.megabytes
  
  belongs_to :topic
  
  has_many   :readings,
             :as => :readable,
             :dependent => :destroy
             
  has_many   :informative_ratings,  
             :dependent => :destroy, 
             :class_name => 'MaterialInformativeRating', 
             :foreign_key => 'subject_id',
             :extend => RatingForUserAssociationExtension

  has_many   :convincing_ratings,  
             :dependent => :destroy, 
             :class_name => 'MaterialConvincingRating', 
             :foreign_key => 'subject_id',
             :extend => RatingForUserAssociationExtension
  
  validates_presence_of :topic, :name
  
  validate :not_both_file_and_url

  validate :presence_of_file_or_url
  validate :attachment_attributes_valid_if_no_url
  validates_presence_of :size, :content_type, :filename, :if => Proc.new { |sup| sup.no_url? }
  # TODO if any of these are missing, only one error should be raised. no?
    
  # can this support material be rated -- depend on the topic it is attached to
  def rateable?
    topic && topic.rateable?
  end
  
  # test that either the file or the url are present  
  def presence_of_file_or_url
    errors.add_to_base("A file or a URL is required") unless filename || !url.blank?
  end
  
  # keep users from creating a support material with both a url AND and file
  def not_both_file_and_url
    errors.add_to_base("Cannot have both a file and a URL") if filename && !url.blank?
  end
  
  # A nice title.
  #
  # returns "file #{filename}" for files or "link {url}" for links 
  def title
    url.blank? ? "file #{File.basename(filename)}" : "link #{url}"
  end
  
  def no_url?
    url.blank?
  end
  
  # a wrapper around acts_as_attachements validation so that it can be made conditional
  def attachment_attributes_valid_if_no_url
    attachment_attributes_valid? if no_url?
  end

  # wrap around attachment_fu's after_destroy callback and make it conditional
  def destroy_file_with_check_first
    return unless filename
    unless File.exists?(full_filename)
      logger.info("File '#{full_filename}' not found for #{self.class} #{id}")
      return    
    end
    destroy_file_without_check_first
  end
  
  alias_method_chain :destroy_file, :check_first

  class << self
    include RuportQueryOverArConnection
    def report(topic_id=nil)
      topic_condition = "WHERE topic_id = #{topic_id}" if topic_id
      report_raw("SELECT id, name, url, filename, topic_id FROM support_materials #{topic_condition}")
    end
  end
  
end
