class AccountController < ApplicationController
  before_filter :bounce_logged_in_users
  skip_before_filter :disable_participation_when_closed, :only => :login
  
  def index
    redirect_to(:action => 'signup') 
  end

  def login
    # @greeting = Snippet.find_or_create_by_name('login_greeting')

    return unless request.post?

    registered_user = RegisteredUser.authenticate(params[:email], params[:password])

    if registered_user
      take_over_with registered_user

      if params[:remember_me] == "1"
        registered_user.remember_me
        send_login_token_cookie
      end

      if params[:refresh_on_login]
        @refresh_on_login = true
        flash[:notice]    = "Logged in successfully"
      end

      respond_to do |wants|
        wants.html do
          flash[:notice] = "Logged in successfully"
          redirect_back_or_default(home_url)
        end
        wants.js
      end
    else
      @message = "Invalid login"
      @message = "Your account has not been verified. Please check your email." if RegisteredUser.authenticate(params[:email], params[:password], false)
      @message = "Your account has been suspended." if RegisteredUser.suspended?(params[:email])
      respond_to do |wants|
        wants.html { flash[:notice] = @message }
        wants.js
      end
    end
  end
  
  def signup
    # @signup_instructions = Snippet.find_or_create_by_name('signup_instructions')
    @user = Participant.new(params[:user])
    
    if ApplicationPolicy::recording_user_numbers?
      @user.user_number = params[:id] if params[:id] && request.get?
    
      # user shows up without an ID
      if request.get? && params[:id].nil?
        render( :action => 'no_user_number') && return
      end
    
      # If a user_number has already been taken, redirect to login with message
      if User.user_number_taken?(@user.user_number)
        flash[:notice] = "You have already registered, please login"
        redirect_to :action => 'login' &&  return
      end
    end
    
    return unless request.post?
    
    if @user.save
      take_over_with @user
      respond_to do |wants|
        wants.html do
          flash[:notice] = "Thanks for registering!"
          redirect_to home_url
        end
        wants.js
      end
    else
      respond_to do |wants|
        wants.html
        wants.js
      end
    end
  end

  def logout
    reset_session
    self.current_user = nil
    cookies.delete :login_token
    flash[:notice] = "You have been logged out."
    # flash[:javascript] = %q{urchinTracker("/funnel/account/logout")}
    redirect_back_or_default('')
  end

  # send an email reminder of the users password
  def remind_password
    return unless request.post?
    user = User.find_by_email(params[:email])
    if user
      UserNotifier.deliver_password_reminder(user)
      flash[:notice] = "Your password has been sent to your email account"
      redirect_to :action => 'login'
    else
      flash[:notice] = "No account was found for that email address"      
    end
  end

  def verify_account
    self.current_user = User.find_by_login_key(params[:key])
    if logged_in? && !current_user.verified?
      current_user.toggle! :verified
      flash[:notice] = "Account Verified"
      # flash[:javascript] = %q{urchinTracker("/funnel/account/verified")}
    end
    redirect_to home_url
  end
    
  protected
  
  
  def bounce_logged_in_users
    if logged_in? && !%w{logout verify_account}.include?(action_name) && current_user.role != :admin
      logger.debug "bouncing a logged in user"  
      redirect_to ''
      return false
    end
  end
  
  # only protect logout action
  def protect?(action)
    %{logout}.include? action
  end
  
  def send_login_token_cookie
    cookies[:login_token]={ :value => "#{current_user.id};#{current_user.reset_login_key!}", :expires => Time.now.utc+1.year }
  end
  
  def take_over_with(new_user)
    previously_held_guest_account = current_user
    self.current_user = new_user
    current_user.take_activity_from_guest_account!(previously_held_guest_account) if previously_held_guest_account
  end
  
end
