;
; Calibrates and plots regional MXD against regional temperature
;
; Define periods for calibration
;
calper=[1881,1960]        ; calibration period
verper=[1961,1994]        ; verification period
thalf=10.
donh=1                    ; 0=do NH 1=do ALL
;
if donh eq 0 then nhtit='NH' else nhtit='ALL'
;
; Get the hemispheric series
;
restore,filename='datastore/compbest_mxd_fixed1950.idlsave'
if donh eq 0 then nhtemp=reform(comptemp(*,3)) $
             else nhtemp=reform(comptemp(*,2))
nhmxd=reform(compmxd(*,2,0))
; Get rid of pre-1871 termperatures
knh=where(timey lt 1871)
nhtemp(knh)=!values.f_nan
;
; Get the extended hemispheric series
;
restore,filename='compts_mxd_fixed1950.idlsave'
fullnhmxd=reform(compmxd(*,2))
;
; Get the extended MXD series
;
restore,filename='regts_mxd_fixed1950.idlsave'
; Gets: timey,bestmxd etc.
yrmxd=timey
fullmxd=bestmxd
; Combine extended hemispheric MXD with these ones
nfull=n_elements(yrmxd)
newfm=fltarr(nfull,nreg+1)*!values.f_nan
newfm(*,0:nreg-1)=fullmxd(*,*)
newfm(*,nreg)=fullnhmxd(*)
fullmxd=newfm
;
; Get regional series (temp, and truncated MXD)
;
restore,filename='datastore/regbest_mxd_fixed1950.idlsave'
; Gets: nreg,regname,regtemp,rawtemp,timey,bestmxd etc.
;
; Identify overlap between full and truncated MXD series
;
nyr=n_elements(timey)
kcomp=where( (yrmxd ge timey(0)) and (yrmxd le timey(nyr-1)) )
;
; Add in the hemispheric series
;
regname=[regname,nhtit]
newrt=fltarr(nyr,nreg+1)*!values.f_nan
newrt(*,0:nreg-1)=regtemp(*,*)
newrt(*,nreg)=nhtemp(*)
regtemp=newrt
newrm=fltarr(nyr,nreg+1)*!values.f_nan
newrm(*,0:nreg-1)=bestmxd(*,*,1)
newrm(*,nreg)=nhmxd(*)
bestmxd=newrm
nreg=nreg+1
;
; Prepare for plotting
;
loadct,39
multi_plot,nrow=3,layout='large',/landscape
if !d.name eq 'X' then begin
  window,xsize=850
  !p.font=-1
endif else begin
  !p.font=0
  device,/helvetica,/bold,font_size=9
endelse
def_1color,10,color='blue'
def_1color,11,color='red'
;
range0=[-2.5,-2.0,-3.5,-2.5,-1.5,-0.5,-2.5,-2.0,-3.0,-1.5]
range1=[ 1.5, 2.0, 2.5, 1.5, 1.0, 1.0, 1.5, 1.5, 1.5, 1.0]
;
calregts=fltarr(nfull,nreg)*!values.f_nan
;
; Process each region separately
;
for ireg = 0 , nreg-1 do begin
  print,regname(ireg)
  ;
  ; Extract series
  ;
  tstemp=reform(regtemp(*,ireg))
  tsmxd=reform(bestmxd(*,ireg))
  tsfull=reform(fullmxd(*,ireg))
  ;
  ; Identify calibration and verification subsets
  ;
  calkl=where( finite(tstemp+tsmxd) and $
               (timey ge calper(0)) and (timey le calper(1)) , ncal )
  verkl=where( finite(tstemp+tsmxd) and $
               (timey ge verper(0)) and (timey le verper(1)) , nver )
  print,'Calibration period:'+string(calper,format='(2I5)')+$
    '  length:'+string(ncal,format='(I5)')
  print,'Verification period:'+string(verper,format='(2I5)')+$
    '  length:'+string(nver,format='(I5)')
  ;
  ; Calibrate the MXD
  ;
  mkcalibrate,tsmxd(calkl),tstemp(calkl),fitcoeff=fitcoeff,autocoeff=autocoeff
  print,'RAW'
  print,'     r   alpha    beta SEalpha  SEbeta    sig    rmse'
  print,fitcoeff,format='(F6.2,4F8.4,F7.2,F8.4)'
  print,' a_mxd a_tem a_resid   ess'
  print,autocoeff,format='(3F6.2,F8.1)'
  ;
  ; And low and high passed too!
  ;
  filter_cru,/nan,thalf,tsin=tsmxd,tshigh=mxdhi,tslow=mxdlo
  filter_cru,/nan,thalf,tsin=tstemp,tshigh=temphi,tslow=templo
  mkcalibrate,mxdhi(calkl),temphi(calkl),fitcoeff=hicoeff,autocoeff=autohi
  print,'HIGH-PASSED',thalf
  print,'     r   alpha    beta SEalpha  SEbeta    sig    rmse'
  print,hicoeff,format='(F6.2,4F8.4,F7.2,F8.4)'
  print,' a_mxd a_tem a_resid   ess'
  print,autohi,format='(3F6.2,F8.1)'
  mkcalibrate,mxdlo(calkl),templo(calkl),fitcoeff=locoeff,autocoeff=autolo,$
    nlag=5
  print,'LOW-PASSED',thalf
  print,'     r   alpha    beta SEalpha  SEbeta    sig    rmse'
  print,locoeff,format='(F6.2,4F8.4,F7.2,F8.4)'
  print,' a_mxd a_tem a_resid   ess'
  print,autolo,format='(3F6.2,F8.1)'
  ;
  ; Generate calibrated record and uncertainties etc.
  ; Use the extended MXD record (after checking it matches the short one!)
  ;
  xxx=tsfull(kcomp)
  yyy=tsmxd
  tse=total( (xxx-yyy)^2 , /nan )
;  print,tse
  if tse gt 0.03 then message,'Series do not match'
  calmxd=tsfull*fitcoeff(2)+fitcoeff(1)
  calregts(*,ireg)=calmxd(*)
  filter_cru,/nan,thalf,tsin=tsfull,tshigh=fullhi,tslow=fulllo
  sepmxd=(fullhi*hicoeff(2)+hicoeff(1))+(fulllo*locoeff(2)+locoeff(1))
  ;
  ; For overlap period with non-missing data, compute correlation between
  ; the temperature series and the 2-band reconstruction
  ;
  kkk=calkl+(timey(0)-yrmxd(0))
  print,'r(Two-band calibrated series,Temp)='+$
    string(correlate(tstemp(calkl),sepmxd(kkk)),format='(F6.2)')
  ;
  if regname(ireg) eq nhtit then !p.multi=[0,1,2,0,0]
  pause
  plot,timey,tstemp,$
    /xstyle,xrange=[1400,2000],xtitle='Year',$
    ytitle='Temperature anomaly  (!Uo!NC wrt 1961-90)',$
    /ystyle,yrange=[range0(ireg),range1(ireg)],$
    title=regname(ireg)
  oplot,yrmxd,calmxd,color=10       ;,thick=3
;  oplot,yrmxd,sepmxd,color=11       ;,thick=3
  oplot,!x.crange,[0.,0.],linestyle=1
  filter_cru,/nan,thalf,tsin=tstemp,tslow=ylow
  oplot,timey,ylow,thick=3
  filter_cru,/nan,thalf,tsin=calmxd,tslow=ylow
  oplot,yrmxd,ylow,thick=3,color=10
;  filter_cru,/nan,thalf,tsin=sepmxd,tslow=ylow
;  oplot,yrmxd,ylow,thick=3,color=11
  ;
  if regname(ireg) eq nhtit then begin
  plot,timey,tstemp,thick=2,$
    /xstyle,xrange=[1850,2000],xtitle='Year',$
    ytitle='Temperature anomaly  (!Uo!NC wrt 1961-90)',$
    title=regname(ireg)
  oplot,yrmxd,calmxd,color=10,thick=3
  oplot,yrmxd,sepmxd,color=11,thick=3
  oplot,!x.crange,[0.,0.],linestyle=1
  endif
  ;
endfor
;
if 0 eq 1 then begin
;
; Now output the calibrated series
;
tempregts=regtemp
tempnyr=nyr
temptimey=timey
nyr=nfull
timey=yrmxd
save,filename='regtemp_calibrated.'+strlowcase(nhtit)+'version.idlsave',$
  nyr,nreg,calregts,regname,timey,tempregts,tempnyr,temptimey
;
missval=-99.9
ml=where(finite(calregts) eq 0,nmiss)
if nmiss gt 0 then calregts(ml)=missval
openw,1,'regtemp_calibrated.dat.'+strlowcase(nhtit)+'version'
printf,1,'Calibrated regional reconstructions using tree-ring-density'
printf,1
printf,1,'Calibrated to give estimates of regional-mean April-September'
printf,1,'temperatures over land areas in some pre-defined regions'
if donh eq 0 then $
  printf,1,'(including region NH which is all land north of 20N).'
printf,1
printf,1,'Skill of the reconstruction generally deteriorates back in time,'
printf,1,'due to fewer chronologies available.'
printf,1
printf,1,missval,'  = missing value'
printf,1,nfull,'  = number of years'
printf,1
printf,1,['Year',regname],format='(A4,10A7)'
for iyr = 0 , nfull-1 do printf,1,yrmxd(iyr),calregts(iyr,*),$
  format='(I4,10F7.2)'
close,1
;
; Also output the series for MATLAB spectral analysis
; Start in 1402 cos 1400-1 are missing in the age-banded multi-PCR series
;
if donh eq 1 then begin
  openw,1,'ts1.matlab_mxdreg_all'
  printf,1,calregts(2:nfull-1,nreg-1),format='(F7.2)'
  close,1
endif
;
endif
;
end
